%%%% A Matlab Code for 2D Non-Penalization SEMDOT Algorithm %%%%
% Example SEMDOT2D(100,100,0.3,4)
function SEMDOT2D(nelx,nely,volfrac,rmin)
nodenrs=reshape(1:(1+nelx)*(1+nely),1+nely,1+nelx);
edofVec=reshape(2*nodenrs(1:end-1,1:end-1)+1,nelx*nely,1);
edofMat=repmat(edofVec,1,8)+repmat([0 1 2*nely+[2 3 0 1] -2 -1],nelx*nely,1);
iK=reshape(kron(edofMat,ones(8,1))',64*nelx*nely,1);
jK=reshape(kron(edofMat,ones(1,8))',64*nelx*nely,1);
%% GEOMETRY & MATERIAL
dx=1; dy=dx; thickness=1; E=1; NU=0.3;
[D0,B,KE]=lk(E,NU,dx,dy,thickness);
%% ACTIVE and PASSIVE
passive=zeros(nely,nelx); passive(1:3/5*nely,2/5*nelx+1:nelx)=1;
%% FILTER
[H,Hs]=HHs(nelx,nely,rmin);
[Hn,Hns]=HnHns(nelx,nely);
%% OPTIMIZATION
sigmax=0.55;
x=repmat(volfrac,nely,nelx); x(find(passive))=1e-3; xPhys=x;
nG=10; ngrid=nG-1;
%% DEFINE LOADS
F=zeros(2*(nelx+1)*(nely+1),1);
nnum=(nely+1)*nelx+3/5*nely+1; nnums=nnum:(nnum+5);
Amplitude=1; F(2*nnums,1)=-Amplitude/length(nnums);
%% DEFINE SUPPORTS
fixednodes=1:(nely+1):(1+2/5*nelx*(nely+1)); fixeddofs=[2*fixednodes-1,2*fixednodes];
alldofs=[1:2*(nelx+1)*(nely+1)]; freedofs=setdiff(alldofs,fixeddofs);
%% PROJECTION
beta=0.5; ER=0.5;
%% STRESS PARA
mu=10; pes=0.4;
%% MMA
m=1; n=nelx*nely; xold1=x(:); xold2=xold1;
xmin=repmat(1e-3,n,1); xmax=ones(n,1); low=xmin; upp=xmax;
a0=1; a=zeros(m,1); cmma=10000*ones(m,1); d=zeros(m,1);
move=0.01; scale_obj=100; scale_cons=1000;
%% ELEMENTAL NODES AND COORDINATES
[nodex,nodey]=meshgrid(0:nelx,0:nely); [fnx,fny]=meshgrid(0:1/ngrid:nelx,0:1/ngrid:nely);
%% CONVERGENCE
epsil=1e-3; maxiter=200;
%% SEMODT
Emin=1e-3; iter=0; change=1;
%% START ITERATION
while (change>epsil || tol>0.001) && iter<maxiter
    iter=iter+1; beta=beta+ER;
    %% FE-ANALYSIS
    sK=reshape(KE(:)*(xPhys(:)'+(1-xPhys(:)')*Emin),64*nelx*nely,1);
    K=sparse(iK,jK,sK); K=(K+K')/2; dK=decomposition(K(freedofs,freedofs));
    U=zeros(2*(nelx+1)*(nely+1),1); U(freedofs,:)=dK\F(freedofs,:);
    EE=xPhys(:)'.^pes; s=(U(edofMat)*(D0*B)').*repmat(EE',1,3);
    vms=sqrt(sum(s.^2,2)-s(:,1).*s(:,2)+2.*s(:,3).^2);
    %% ADJOINT
    sumfun=0;
    RightplT=sparse(2*(nelx+1)*(nely+1),1); dereledbu=sparse(nelx*nely,1);
    for e=1:nelx*nely
        edof=edofMat(e,:)'; xe=xPhys(e); stress=s(e,:);
        Mespersigma(1)=(2*stress(1)-stress(2))/2/vms(e);
        Mespersigma(2)=(2*stress(2)-stress(1))/2/vms(e);
        Mespersigma(3)=3*stress(3)/vms(e);
        stress_ratio=vms(e)/sigmax;
        sumfun=sumfun+stress_ratio^mu;
        dereledbu(e)=Mespersigma*pes*xe^(pes-1)*D0*B*U(edofMat(e,:));
        RightplT(edof')=RightplT(edof')+(stress_ratio^(mu-1)/sigmax)*B'*D0'*Mespersigma';
    end
    Obj(iter,1)=sumfun^(1/mu); Con(iter,1)=sum(xPhys(:))/nelx/nely;
    %% SENSITIVITY
    RightT=RightplT*sumfun^(1/mu-1);
    lamd=zeros(2*(nelx+1)*(nely+1),1); lamd(freedofs,:)=dK\RightT(freedofs,:);
    dc=zeros(nelx*nely,1);
    for e=1:nelx*nely
        edof=edofMat(e,:)'; Ue=U(edof); lamde=lamd(edof); xe=xPhys(e);
        dKE=KE; stress_ratio=vms(e)/sigmax;
        dc(e)=(stress_ratio^(mu-1)/sigmax*sumfun^(1/mu-1))*dereledbu(e)-xe^pes*lamde'*dKE*Ue;
    end
    dc=(xPhys(:)+(1-xPhys(:))*Emin).*dc;
    dm=ones(nely,nelx)/nelx/nely;
    %% FILTERING
    dc=reshape(dc,nely,nelx); dc(:)=H*(dc(:)./Hs);
    dm(:)=H*(dm(:)./Hs);
    %% MMA OBJ % CONS
    f0val=scale_obj*Obj(end);
    df0dx=scale_obj*dc(:);
    fval(1,1)=scale_cons*(Con(end)/volfrac-1);
    dfdx(1,:)=scale_cons*dm(:)'/volfrac;
    %% MMA SOLVER
    dv=x(:); xmax=min(1,dv+move); xmin=max(0,dv-move);
    [xmma,~,~,~,~,~,~,~,~,low,upp]=...
        mmasub(m,n,iter,dv,xmin,xmax,xold1,xold2,f0val,df0dx,fval,dfdx,low,upp,a0,a,cmma,d);
    if iter>2, xold2=xold1; xold1=dv; elseif iter>1, xold1=dv; end
    x=reshape(xmma,nely,nelx);
    x(find(passive))=1e-3;
    xnew=reshape(xmma,nely,nelx);
    xPhys(:)=(H*x(:))./Hs; xPhys(find(passive))=1e-3;
    %% UPDATE POINT DESNIGY BY A HEAVISIDE SMOOTH/STEP FUNCTION
    [xg,ls,xPhys,tol]=xg_xPhys(nelx,nely,xPhys,Hn,Hns,nodex,nodey,fnx,fny,ngrid,beta);
    %% CHECK CONVERGENCE
    change=sum(abs(xnew(:)-x(:)))/(volfrac*nely*nelx);
    x=xnew;
    %% PLOT RESULTS
    fprintf('It.:%3i Obj.:%6.3f Vol.:%4.3f tol.:%4.3f chan.:%4.3f beta.:%g.\n',iter,Obj(end),Con(end),tol,change,beta);
    contourf(fnx, flipud(fny), xg-ls, [0 0]); axis equal tight off; drawnow
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This code was written by:                                               %
%                                                                         %
% Dr. Yun-Fei Fu                                                          %
% Department of Mechanical Engineering, University of Alberta, Canada     %
%                                                                         %
% Prof. Bernard Rolfe                                                     %
% School of Engineering, Faculty of Science Engineering and               %
% Built Environment, Deakin University, Australia                         %
%                                                                         %
% Please send your comments and questions to: jackyunfei.fu@outlook.com   %
%                                                                         %
% The code is intended for educational purposes and theoretical details   %
% are discussed in the book: "Smooth Topological Design of Continuum      %
% Structures", YF Fu and B Rolfe, CRC press.                              %
%                                                                         %
%                                                                         %
% Disclaimer:                                                             %
% The authors do not guarantee that the code is free from errors.         %
% Furthermore, the authors shall not be liable in any event caused by the %
% use of the program.                                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%